#include "Network.h"
#include "Control.h"
#include <stdlib.h>
#include <math.h>
#include <stdio.h>



CNetwork::CNetwork(CControl *pControl){
	mpControl = pControl;
	mnLayers = 3;
	mpLayerSizes = (int*)malloc(mnLayers * sizeof(int));
	mpLayerSizes[0] = 64;
	mpLayerSizes[1] = 2;
	mpLayerSizes[2] = 1;
	mLearningRate = 1.0f;
	mnCorrect = mnWrong = 0;

	int layer;
	mdpNeurons = (CNeuron**)malloc(mnLayers * sizeof(CNeuron*));
	for(layer = 0;  layer < mnLayers;  ++layer){
		mdpNeurons[layer] = (CNeuron*)malloc(mpLayerSizes[layer] * sizeof(CNeuron));
		for(int neuron = 0;  neuron < mpLayerSizes[layer];  ++neuron)
			mdpNeurons[layer][neuron].mOutput = mdpNeurons[layer][neuron].mError = 0.0f;
	}

	mtpLinks = (CLink***)malloc(mnLayers * sizeof(CLink**));
	for(layer = 1;  layer < mnLayers;  ++layer){
		mtpLinks[layer] = (CLink**)malloc(mpLayerSizes[layer] * sizeof(CLink*));
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron){
			mtpLinks[layer][postNeuron] = (CLink*)malloc(mpLayerSizes[layer-1] * sizeof(CLink));
			for(int preNeuron = 0;  preNeuron < mpLayerSizes[layer-1];  ++preNeuron)
				mtpLinks[layer][postNeuron][preNeuron].mWeight = RangedRandom(-0.15f, 0.15f);
		}
	}
}



CNetwork::~CNetwork(){
	int layer;
	for(layer = 0;  layer < mnLayers;  ++layer)
		free(mdpNeurons[layer]);
	free(mdpNeurons);
	for(layer = 1;  layer < mnLayers;  ++layer){
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron)
			free(mtpLinks[layer][postNeuron]);
		free(mtpLinks[layer]);
	}
	free(mtpLinks);
	free(mpLayerSizes);
}



void CNetwork::ForwardPropagate(float *pInputs){
	for(int input = 0;  input < mpLayerSizes[0];  ++input)
		mdpNeurons[0][input].mOutput = pInputs[input];
	for(int layer = 1;  layer < mnLayers;  ++layer)
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron){
			float activation = 0.0f;
			for(int preNeuron = 0;  preNeuron < mpLayerSizes[layer-1];  ++preNeuron)
				activation += mdpNeurons[layer-1][preNeuron].mOutput * mtpLinks[layer][postNeuron][preNeuron].mWeight;
			mdpNeurons[layer][postNeuron].mOutput = 1.0f / (1.0f + (float)exp(-activation));
		}
}



void CNetwork::JudgeOutputs(float *pCorrectOutputs){
	for(int output = 0;  output < mpLayerSizes[mnLayers-1];  ++output){
		mdpNeurons[mnLayers-1][output].mError = pCorrectOutputs[output] - mdpNeurons[mnLayers-1][output].mOutput;
		bool bCorrect = pCorrectOutputs[output] == 1.0f;
		bool bNetwork = mdpNeurons[mnLayers-1][output].mOutput > 0.5f;
		if(bCorrect && bNetwork  ||  !bCorrect && !bNetwork)  ++mnCorrect;
		else  ++mnWrong;
	}
}



void CNetwork::BackwardPropagate(){
	for(int layer = mnLayers-1;  layer >= 1;  --layer)
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron){
			CNeuron *pPost = &mdpNeurons[layer][postNeuron];
			pPost->mError = pPost->mOutput * (1.0f - pPost->mOutput) * pPost->mError;
			for(int preNeuron = 0;  preNeuron < mpLayerSizes[layer-1];  ++preNeuron){
				CNeuron *pPre = &mdpNeurons[layer-1][preNeuron];
				CLink *pLink = &mtpLinks[layer][postNeuron][preNeuron];
				pLink->mWeight += pPost->mError * pPre->mOutput * mLearningRate;
				pPre->mError += pPost->mError * pLink->mWeight;
			}
			pPost->mError = 0.0f;
		}
}



float CNetwork::RangedRandom(float low, float high){
	float range = high - low;
	return low + rand()/(float)RAND_MAX * range;
}



void CNetwork::PrintStatistics(){
	float ratioCorrect = mnCorrect/(float)(mnCorrect+mnWrong);
	printf("cycle %3d: %.4fcorrect\n", mpControl->mCycle, ratioCorrect);
	mnCorrect = mnWrong = 0;
}



float CNetwork::GetAndResetCorrect(){
	float ratioCorrect = mnCorrect/(float)(mnCorrect+mnWrong);
	mnCorrect = mnWrong = 0;
	return ratioCorrect;
}
