#include "Network.h"
#include "Control.h"
#include <stdlib.h>
#include <math.h>
#include <stdio.h>



CNetwork::CNetwork(CControl *pControl){
	mpControl = pControl;

	// FORCE-FEED THE NETWORK SETUP.
	mnLayers = 3;
	mpLayerSizes = (int*)malloc(mnLayers * sizeof(int));
	mpLayerSizes[0] = 64;
	mpLayerSizes[1] = 2;
	mpLayerSizes[2] = 1;
	mLearningRate = 1.0f;
	mnCorrect = mnWrong = 0;

	// ALLOCATE THE NEURONS.
	int layer;
	mdpNeurons = (CNeuron**)malloc(mnLayers * sizeof(CNeuron*));  // First we make mdpNeurons point to its array of child "nodes", which each represent a layer.
	for(layer = 0;  layer < mnLayers;  ++layer){  // Then, for each layer...
		mdpNeurons[layer] = (CNeuron*)malloc(mpLayerSizes[layer] * sizeof(CNeuron));  // we make that layer node point to an array of neuron nodes.
		for(int neuron = 0;  neuron < mpLayerSizes[layer];  ++neuron)  // Then, for each neuron in the current layer...
			mdpNeurons[layer][neuron].mOutput = mdpNeurons[layer][neuron].mError = 0.0f;  // we initialize its output and error.
	}

	// ALLOCATE THE LINKS.
	mtpLinks = (CLink***)malloc(mnLayers * sizeof(CLink**));
	for(layer = 1;  layer < mnLayers;  ++layer){  // Note we start at layer 1 for the links since layer 0 has no links pointing back from its neurons.
		mtpLinks[layer] = (CLink**)malloc(mpLayerSizes[layer] * sizeof(CLink*));
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron){  // The "post" neuron is the neuron that comes after a link.
			mtpLinks[layer][postNeuron] = (CLink*)malloc(mpLayerSizes[layer-1] * sizeof(CLink));  // There's one link going back to each neuron in the previous layer.
			for(int preNeuron = 0;  preNeuron < mpLayerSizes[layer-1];  ++preNeuron)
				mtpLinks[layer][postNeuron][preNeuron].mWeight = RangedRandom(-0.15f, 0.15f);  // Setting each link to an initially random weight is critical for learning.
		}
	}
}



CNetwork::~CNetwork(){
	int layer;
	for(layer = 0;  layer < mnLayers;  ++layer)
		free(mdpNeurons[layer]);
	free(mdpNeurons);
	for(layer = 1;  layer < mnLayers;  ++layer){
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron)
			free(mtpLinks[layer][postNeuron]);
		free(mtpLinks[layer]);
	}
	free(mtpLinks);
	free(mpLayerSizes);
}



void CNetwork::ForwardPropagate(float *pInputs){
	/* COPY THE INPUTS INTO THE NETWORK
	To do this we simply set the output of the neurons in the first layer to be the patterns inputs.  So, the first layer's neurons
	actually act like dummy neurons: they never do any processing, they're just there to simulate "sensory input".
	*/
	for(int input = 0;  input < mpLayerSizes[0];  ++input)
		mdpNeurons[0][input].mOutput = pInputs[input];

	// PROPAGATE ACTIVATIONS THROUGH THE NETWORK.
	for(int layer = 1;  layer < mnLayers;  ++layer)  // For each layer, starting with layer 1 (not the input layer)...
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron){  // and for each neuron in that layer...
			float activation = 0.0f;  // initialize its activation to 0.
			for(int preNeuron = 0;  preNeuron < mpLayerSizes[layer-1];  ++preNeuron)  // Then, consider each link leading into that neuron from the neurons in the previous layer;
				activation += mdpNeurons[layer-1][preNeuron].mOutput * mtpLinks[layer][postNeuron][preNeuron].mWeight;  // and, add to out activation the output of the neuron times the weight on the link connecting us to it.
			/* The total activation could be a large or small, positive or negative number.  So, we make our neuron's final output
			constrained between 0 and 1 by using the following function.  There is no "magic" in the function, any function that
			compresses a wide range of numbers to a value between 0 and 1 would work pretty good. */
			mdpNeurons[layer][postNeuron].mOutput = 1.0f / (1.0f + (float)exp(-activation));
		}
}



void CNetwork::JudgeOutputs(float *pCorrectOutputs){
	for(int output = 0;  output < mpLayerSizes[mnLayers-1];  ++output){  // For each neuron in the output layer...
		/* set its "gross" error to the correct output minus its output.  Thus, if the correct output was 1.0 and the neuron
		outputted 0.7 then the gross error would be +0.3, which indicates we want the neuron's output to move in the positive
		direction.  If the correct output was 0.0 then our gross error would be -0.7. */
		mdpNeurons[mnLayers-1][output].mError = pCorrectOutputs[output] - mdpNeurons[mnLayers-1][output].mOutput;

		// COLLECT SIMPLE STATISTICS.
		bool bCorrect = pCorrectOutputs[output] == 1.0f;
		bool bNetwork = mdpNeurons[mnLayers-1][output].mOutput > 0.5f;
		if(bCorrect && bNetwork  ||  !bCorrect && !bNetwork)  ++mnCorrect;
		else  ++mnWrong;
	}
}



void CNetwork::BackwardPropagate(){
	// Note that JudgeOutputs should be called just before this so that the gross errors in the output layer are set.
	for(int layer = mnLayers-1;  layer >= 1;  --layer)  // Starting at the LAST layer and going back...
		for(int postNeuron = 0;  postNeuron < mpLayerSizes[layer];  ++postNeuron){  // for each neuron in that layer...
			CNeuron *pPost = &mdpNeurons[layer][postNeuron];  // make a shortcut to that neuron to simplify the following code,
			/* and convert its gross error into "output-sensitive error".  The factor output*(1-output) reduces a neuron's error
			when its output was very close to 1 or very close to 0.  This is when the neuron is more "sure".  Since the error is
			less, the neuron is adjusted less.  This promotes the network to settle down into a particular state. */
			pPost->mError = pPost->mOutput * (1.0f - pPost->mOutput) * pPost->mError;
			for(int preNeuron = 0;  preNeuron < mpLayerSizes[layer-1];  ++preNeuron){  // Then propagate our error to each neuron in the previous layer.
				CNeuron *pPre = &mdpNeurons[layer-1][preNeuron];  // Make a shortcut for the previous neuron and...
				CLink *pLink = &mtpLinks[layer][postNeuron][preNeuron];  // make a shortcut for the link going to that neuron.
				/* What is happening really is that now that we know the error that the postNeuron caused in the final
				result, we need to account for how much each previous neuron/link contributed to that error.  We pass
				back the error to the previous neuron in proportion to the weight on the link connecting us to them.
				Then we adjust the weight such that the previous neuron's output will tend to cause the postNeuron's
				output to be closer to the goal. */
				pPre->mError += pPost->mError * pLink->mWeight;
				pLink->mWeight += pPost->mError * pPre->mOutput * mLearningRate;
			}
			pPost->mError = 0.0f;  // Re-set the error to 0 for the next backpropagation.
		}
}



float CNetwork::RangedRandom(float low, float high){
	float range = high - low;
	return low + rand()/(float)RAND_MAX * range;
}



void CNetwork::PrintStatistics(){
	float ratioCorrect = mnCorrect/(float)(mnCorrect+mnWrong);
	printf("cycle %3d: %.4fcorrect\n", mpControl->mCycle, ratioCorrect);
	mnCorrect = mnWrong = 0;
}



float CNetwork::GetAndResetCorrect(){
	float ratioCorrect = mnCorrect/(float)(mnCorrect+mnWrong);
	mnCorrect = mnWrong = 0;
	return ratioCorrect;
}
